/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  LibtoolVersion,
  TranslatedString,
  WalletCoreVersion
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { Checkbox } from "../components/Checkbox.js";
import { EnabledBySettings } from "../components/EnabledBySettings.js";
import { Part } from "../components/Part.js";
import { SelectList } from "../components/SelectList.js";
import {
  Input,
  SubTitle,
  WarningBox
} from "../components/styled/index.js";
import { useAlertContext } from "../context/alert.js";
import { useBackendContext } from "../context/backend.js";
import { useAsyncAsHook } from "../hooks/useAsyncAsHook.js";
import { useBackupDeviceName } from "../hooks/useBackupDeviceName.js";
import { useSettings } from "../hooks/useSettings.js";
import { ToggleHandler } from "../mui/handlers.js";
import { Settings } from "../platform/api.js";
import { platform } from "../platform/foreground.js";
import { WALLET_CORE_SUPPORTED_VERSION } from "../wxApi.js";

const GIT_HASH = typeof __GIT_HASH__ !== "undefined" ? __GIT_HASH__ : undefined;

export function SettingsPage(): VNode {
  const [settings, updateSettings] = useSettings();
  const { safely } = useAlertContext();
  const { name, update } = useBackupDeviceName();
  const webex = platform.getWalletWebExVersion();
  const api = useBackendContext();

  const hook = useAsyncAsHook(async () => {
    const version = await api.wallet.call(WalletApiOperation.GetVersion, {});
    return { version };
  });

  const version = hook && !hook.hasError ? hook.response.version : undefined

  return (
    <SettingsView
      deviceName={name}
      setDeviceName={update}
      autoOpenToggle={{
        value: settings.autoOpen,
        button: {
          onClick: safely("update support injection", async () => {
            updateSettings("autoOpen", !settings.autoOpen);
          }),
        },
      }}
      advanceToggle={{
        value: settings.advancedMode,
        button: {
          onClick: safely("update advance mode", async () => {
            updateSettings("advancedMode", !settings.advancedMode);
          }),
        },
      }}
      langToggle={{
        value: settings.langSelector,
        button: {
          onClick: safely("update lang selector", async () => {
            updateSettings("langSelector", !settings.langSelector);
          }),
        },
      }}
      webexVersion={{
        version: webex.version,
        hash: GIT_HASH,
      }}
      coreVersion={version}
    />
  );
}

export interface ViewProps {
  deviceName: string;
  setDeviceName: (s: string) => Promise<void>;
  autoOpenToggle: ToggleHandler;
  advanceToggle: ToggleHandler;
  langToggle: ToggleHandler;
  coreVersion: WalletCoreVersion | undefined;
  webexVersion: {
    version: string;
    hash: string | undefined;
  };
}

export function SettingsView({
  autoOpenToggle,
  advanceToggle,
  langToggle,
  coreVersion,
  webexVersion,
}: ViewProps): VNode {
  const { i18n, lang, supportedLang, changeLanguage } = useTranslationContext();

  const api = useBackendContext();

  return (
    <Fragment>
      <section>
        <SubTitle>
          <i18n.Translate>Navigator</i18n.Translate>
        </SubTitle>
        <Checkbox
          label={i18n.str`Automatically open wallet`}
          name="autoOpen"
          description={
            <i18n.Translate>
              Open the wallet when a payment action is found.
            </i18n.Translate>
          }
          enabled={autoOpenToggle.value!}
          onToggle={autoOpenToggle.button.onClick!}
        />

        <SubTitle>
          <i18n.Translate>Version Info</i18n.Translate>
        </SubTitle>
        <Part
          title={i18n.str`Web Extension`}
          text={
            <span>
              {webexVersion.version}{" "}
              <EnabledBySettings name="advancedMode">
                {webexVersion.hash}
              </EnabledBySettings>
            </span>
          }
        />
        {coreVersion && (
          <Fragment>
            {LibtoolVersion.compare(
              coreVersion.version,
              WALLET_CORE_SUPPORTED_VERSION,
            )?.compatible ? undefined : (
              <WarningBox>
                <i18n.Translate>
                  The version of wallet core is not supported. (supported
                  version: {WALLET_CORE_SUPPORTED_VERSION}, wallet version: {coreVersion.version})
                </i18n.Translate>
              </WarningBox>
            )}
            <EnabledBySettings name="advancedMode">
              <Part
                title={i18n.str`Exchange compatibility`}
                text={<span>{coreVersion.exchange}</span>}
              />
              <Part
                title={i18n.str`Merchant compatibility`}
                text={<span>{coreVersion.merchant}</span>}
              />
              <Part
                title={i18n.str`Bank compatibility`}
                text={<span>{coreVersion.bank}</span>}
              />
              <Part
                title={i18n.str`Wallet Core compatibility`}
                text={<span>{coreVersion.version}</span>}
              />
            </EnabledBySettings>
          </Fragment>
        )}
        <SubTitle>
          <i18n.Translate>Settings</i18n.Translate>
        </SubTitle>
        <Checkbox
          label={i18n.str`Enable developer mode`}
          name="devMode"
          description={i18n.str`Show more information and options in the UI`}
          enabled={advanceToggle.value!}
          onToggle={advanceToggle.button.onClick!}
        />
        <EnabledBySettings name="advancedMode">
          <AdvanceSettings />
        </EnabledBySettings>
        <EnabledBySettings name="langSelector">
          <SubTitle>
            <i18n.Translate>Display</i18n.Translate>
          </SubTitle>
          <Input>
            <SelectList
              label={<i18n.Translate>Current Language</i18n.Translate>}
              list={supportedLang}
              name="lang"
              value={lang}
              onChange={(v) => changeLanguage(v)}
            />
          </Input>
        </EnabledBySettings>
      </section>
    </Fragment>
  );
}

type Info = { label: TranslatedString; description: TranslatedString };
type Options = {
  [k in keyof Settings]?: Info;
};
function AdvanceSettings(): VNode {
  const [settings, updateSettings] = useSettings();
  const api = useBackendContext();
  const { i18n } = useTranslationContext();
  const o: Options = {
    backup: {
      label: i18n.str`Show backup feature`,
      description: i18n.str`Backup integration still in beta.`,
    },
    p2p_aliases: {
      label: i18n.str`Show P2P alias feature`,
      description: i18n.str`P2P aliases integration still in beta.`,
    },
    suspendIndividualTransaction: {
      label: i18n.str`Show suspend/resume transaction`,
      description: i18n.str`Prevent transaction from doing network request.`,
    },
    showRefeshTransactions: {
      label: i18n.str`Show refresh transaction type in the transaction list`,
      description: i18n.str`Refresh transaction will be hidden by default if the refresh operation doesn't have fee.`,
    },
    extendedAccountTypes: {
      label: i18n.str`Show more account types on deposit`,
      description: i18n.str`Extends the UI to more payment target types.`,
    },
    showJsonOnError: {
      label: i18n.str`Show JSON on error`,
      description: i18n.str`Print more information about the error. Useful for debugging.`,
    },
    walletAllowHttp: {
      label: i18n.str`Allow HTTP connections`,
      description: i18n.str`Using HTTP connection may be faster but unsafe (wallet restart required)`,
    },
    langSelector: {
      label: i18n.str`Lang selector`,
      description: i18n.str`Allows to manually change the language of the UI. Otherwise it will be automatically selected by your browser configuration.`,
    },
    showExchangeManagement: {
      label: i18n.str`Edit exchange management`,
      description: i18n.str`Allows to see the list of exchange, remove, add and switch before withdrawal.`,
    },
    selectTosFormat: {
      label: i18n.str`Select terms of service format`,
      description: i18n.str`Allows to render the terms of service on different format selected by the user.`,
    },
    showWalletActivity: {
      label: i18n.str`Show wallet activity`,
      description: i18n.str`Show the wallet notification and observability event in the UI.`,
    },
    walletEnableV1Contracts: {
      label: i18n.str`Enable v1 contract`,
      description: i18n.str`v1 is under development and wallet-core converts v1 contract to v0. Enable this options for testing`,
    }
  };
  return (
    <Fragment>
      <section>
        {Object.entries(o).map(([name, { label, description }]) => {
          const settingsName = name as keyof Settings;
          return (
            <Checkbox
              label={label}
              name={name}
              key={name}
              description={description}
              enabled={settings[settingsName]}
              onToggle={async () => {
                updateSettings(settingsName, !settings[settingsName]);
                await api.background.call("reinitWallet", undefined);
              }}
            />
          );
        })}
      </section>
    </Fragment>
  );
}
