/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Christian Blättler
 */

import {
  HttpStatusCode,
  TalerError,
  TalerMerchantApi,
  assertUnreachable,
} from "@gnu-taler/taler-util";
import {
  LocalNotificationBannerBulma,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { VNode, h } from "preact";
import { useState } from "preact/hooks";
import { ErrorLoadingMerchant } from "../../../../components/ErrorLoadingMerchant.js";
import { Loading } from "../../../../components/exception/loading.js";
import { ConfirmModal } from "../../../../components/modal/index.js";
import { useSessionContext } from "../../../../context/session.js";
import { useInstanceTokenFamilies } from "../../../../hooks/tokenfamily.js";
import { LoginPage } from "../../../login/index.js";
import { NotFoundPageOrAdminCreate } from "../../../notfound/index.js";
import { CardTable } from "./Table.js";

const TALER_SCREEN_ID = 68;

interface Props {
  onUnauthorized: () => VNode;
  onNotFound: () => VNode;
  onCreate: () => void;
  onSelect: (slug: string) => void;
}
export default function TokenFamilyList({ onCreate, onSelect }: Props): VNode {
  const result = useInstanceTokenFamilies();

  const [deleting, setDeleting] =
    useState<TalerMerchantApi.TokenFamilySummary | null>(null);

  const { i18n } = useTranslationContext();
  const { state: session, lib } = useSessionContext();
  const [notification, safeFunctionHandler] = useLocalNotificationBetter();

  if (!result) return <Loading />;
  if (result instanceof TalerError) {
    return <ErrorLoadingMerchant error={result} />;
  }
  if (result.type === "fail") {
    switch (result.case) {
      case HttpStatusCode.NotFound: {
        return <NotFoundPageOrAdminCreate />;
      }
      case HttpStatusCode.Unauthorized: {
        return <LoginPage />;
      }
      default: {
        assertUnreachable(result);
      }
    }
  }

  const remove = safeFunctionHandler(
    lib.instance.deleteTokenFamily.bind(lib.instance),
    !session.token || !deleting ? undefined : [session.token, deleting.slug],
  );
  remove.onSuccess = () => {
    setDeleting(null);
    return i18n.str`Token family has been deleted.`;
  };
  remove.onFail = (fail) => {
    switch (fail.case) {
      case HttpStatusCode.Unauthorized:
        return i18n.str`Unauthorized`;
      case HttpStatusCode.NotFound:
        return i18n.str`Not found`;
    }
  };

  return (
    <section class="section is-main-section">
      <LocalNotificationBannerBulma notification={notification} />

      <CardTable
        instances={result.body.token_families}
        onCreate={onCreate}
        onSelect={(tokenFamily) => onSelect(tokenFamily.slug)}
        onDelete={(fam) => setDeleting(fam)}
      />

      {deleting && (
        <ConfirmModal
          label={i18n.str`Delete token family`}
          description={i18n.str`Delete the token family "${deleting.name}"`}
          danger
          active
          onCancel={() => setDeleting(null)}
          confirm={remove}
        >
          <p>
            <i18n.Translate>
              If you delete the <b>&quot;{deleting.name}&quot;</b> token family
              (Slug: <b>{deleting.slug}</b>), all issued tokens will become
              invalid.
            </i18n.Translate>
          </p>
          <p class="warning">
            <i18n.Translate>
              Deleting a token family{" "}
              <b>
                <i18n.Translate>can't be undone</i18n.Translate>
              </b>
              .
            </i18n.Translate>
          </p>
        </ConfirmModal>
      )}
    </section>
  );
}
